//  wminterface.cc for fluxter.
//
//  Copyright (c) 2002 Steve Cooper, stevencooper@isomedia.com
//  Copyright (c) 1998-2000 by John Kennis, jkennis@chello.nl
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
//
// (See the included file COPYING / GPL-2.0)
//

#include "wminterface.hh"
#include "resource.hh"

WMInterface::WMInterface(ToolWindow *toolwindow) {
    timer=new BTimer(toolwindow->getCurrentScreenInfo()->getBaseDisplay(),this) ;
    timer->setTimeout(1000);
    timer->fireOnce(False);
    timer->start();
    bbtool=toolwindow;
}

WMInterface::~WMInterface() {}

void WMInterface::moduleInit() {
}

void WMInterface::sendClientMessage(Atom atom, XID data) {
    XEvent e;
    unsigned long mask;

#ifdef DEBUG
    fprintf(stderr,"Attempting to send message...\n");
#endif
    e.xclient.type = ClientMessage;
    e.xclient.window =  bbtool->getCurrentScreenInfo()->getRootWindow();
  
//bbtool->framewin;

    e.xclient.message_type = atom;
    e.xclient.format = 32;
    e.xclient.data.l[0] = (unsigned long) data;
    e.xclient.data.l[1] = 0;

    mask =  SubstructureRedirectMask;
    XSendEvent(bbtool->getXDisplay(), 
               bbtool->getCurrentScreenInfo()->getRootWindow(),
               False, mask, &e);
}

int WMInterface::getCurrentDesktop(void) {
    Atom type_ret;
    int format_ret;
    unsigned long nitems_ret, unused;
    unsigned long *data = 0;
    int res = -1;

    if (XGetWindowProperty(bbtool->getXDisplay(),
                           bbtool->getCurrentScreenInfo()->getRootWindow(),
                           bbtool->getNETCurrentDesktopAtom(),
                           0l, 1L, False, 0,
                           &type_ret,
                           &format_ret, &nitems_ret,
                           &unused,
                           (unsigned char **) &data ) == Success) {
        if (nitems_ret == 1) {
            res = data[0];
            XFree((void *) data);
        }
    }
    return res;
}

void WMInterface::changeDesktop(int desk_number) {
    unsigned long int data=desk_number;

    sendClientMessage(bbtool->getNETCurrentDesktopAtom(),data);
}


void WMInterface::sendWindowToDesktop(Window win,int desk_number) {
    XEvent e;
    unsigned long mask;

    e.xclient.type = ClientMessage;
    e.xclient.window = win;
    e.xclient.message_type = bbtool->getNETWMDesktopAtom();
    e.xclient.format = 32;
    e.xclient.data.l[0] = desk_number;
    e.xclient.data.l[1] = 2;
    e.xclient.data.l[2] = 0;
    e.xclient.data.l[3] = 0; 
    e.xclient.data.l[4] = 0;
    e.xclient.data.l[5] = 0;

    mask =  SubstructureRedirectMask;
    XSendEvent(bbtool->getXDisplay(), 
               bbtool->getCurrentScreenInfo()->getRootWindow(),
               False, mask, &e);

}

void WMInterface::setWindowFocus(Window win) {
    XEvent e;
    unsigned long mask;

    e.xclient.type = ClientMessage;
    e.xclient.window =  win;

    e.xclient.message_type = bbtool->getNETActiveWindowAtom();
    e.xclient.format = 32;
    e.xclient.data.l[0] = 2;
    e.xclient.data.l[1] = 0;
    e.xclient.data.l[2] = 0;

    mask =  SubstructureRedirectMask;
    XSendEvent(bbtool->getXDisplay(), 
               bbtool->getCurrentScreenInfo()->getRootWindow(),
               False, mask, &e);
}

bool WMInterface::isIconicState(Window win) {
    return hasWindowAtom(win,bbtool->getNETWMStateHiddenAtom());
}

bool WMInterface::hasFocus(Window win) {
    Atom type_ret;;
    int format_ret;
    unsigned long nitems_ret, unused;
    unsigned long *data = 0;
    bool res = False;

    if (XGetWindowProperty(bbtool->getXDisplay(),
                           bbtool->getCurrentScreenInfo()->getRootWindow(),
                           bbtool->getNETActiveWindowAtom(),
                           0l, 1L, False, 0,
                           &type_ret,
                           &format_ret, &nitems_ret,
                           &unused,
                           (unsigned char **) &data ) == Success) {
        if (nitems_ret == 1) res = (data[0] == win);
        XFree((void *) data);
    }
    return res;
}


bool WMInterface::hasWindowAtom(Window win, Atom atom)
{
    Atom type_ret = atom;
    int format_ret;
    unsigned long nitems_ret, unused;
    unsigned long *data = 0;

    bool res = false;

    if (XGetWindowProperty(bbtool->getXDisplay(),
                           win, bbtool->getNETWMStateAtom(),
                           0l, 10L, False, 0, &type_ret,
                           &format_ret, &nitems_ret,
                           &unused,
                           (unsigned char **) &data ) == Success
        ){
        if (data != None){
            for (uint i = 0; i < nitems_ret; i++){
                if (data[i] == atom){
                    res = true;
                    break;
                }
            }
        }
        XFree((void *) data);
    }
    return res;

}

int WMInterface::getNumberOfDesktops(void) {
    Atom type_ret;
    int format_ret;
    unsigned long nitems_ret, unused;
    unsigned long *data = 0;

    int res = -1;

    if (XGetWindowProperty(bbtool->getXDisplay(),
                           bbtool->getCurrentScreenInfo()->getRootWindow(),
                           bbtool->getNETNumberOfDesktopsAtom(),
                           0l, 1L, False, 0, &type_ret,
                           &format_ret, &nitems_ret,
                           &unused,
                           (unsigned char **) &data ) == Success) {
        if ( data != None ) res =  static_cast<int> (data[0]);
        XFree((void *) data);

    }
    return res;
}

const vector<Window>& WMInterface::getClients(void)
{
    Atom type_ret;
    int format_ret;
    unsigned long nitems_ret, unused;
    unsigned long *data = 0;

    winvec.clear();
    
    // get current set of windows
    if (XGetWindowProperty(bbtool->getXDisplay(),
                           bbtool->getCurrentScreenInfo()->getRootWindow(),
                           bbtool->getNETClientListStackingAtom(),
                           0l, 1000, False, XA_WINDOW, &type_ret,
                           &format_ret, &nitems_ret,
                           &unused,(unsigned char **) &data ) == Success)
    {

        if (type_ret == XA_WINDOW && format_ret == 32) {
            Window *wins = (Window *) data;
            for (unsigned int i=0; i < nitems_ret; i++) {
                winvec.push_back(wins[i]);
            }
        }
        XFree((void *) data);
    }
    return winvec;
}   
/**
 *
 * @param win
 * @return index of the desktop (workspace) a window in
 */
int WMInterface::getWindowWorkspaceID(Window win) {

    Atom type_ret;
    int format_ret;
    unsigned long nitems_ret, unused;
    unsigned long *data = 0;

    int res = -1;

    if (XGetWindowProperty(bbtool->getXDisplay(),
                           win, bbtool->getNETWMDesktopAtom(),
                           0l, 10L, False, XA_CARDINAL, &type_ret,
                           &format_ret, &nitems_ret,
                           &unused,
                           (unsigned char **) &data ) == Success
        ){
        if ( data != None )
            res = static_cast<int>(*data);
        XFree((void *) data);
    }
    return res;
}

void WMInterface::addSticky(WindowList *tmp)
{
    int i=0;
        
    tmp->sticky=True;
        
    LinkedListIterator<DesktopList> desktop_it(bbtool->desktopList);

    for (; desktop_it.current(); desktop_it++) {
        i++;

        if (bbtool->getCurrentDesktopNr()!=
            desktop_it.current()->desktop_nr) {
            WindowList *copy = new WindowList;
            copy->win= tmp->win;
            copy->width=tmp->width;
            copy->height=tmp->height;
            copy->x_position=tmp->x_position;
            copy->y_position=tmp->y_position;
            copy->icon=tmp->icon;
            copy->sticky=tmp->sticky;
            copy->focused=tmp->focused;
            copy->shaded = tmp->shaded;
            copy->desktop_nr=desktop_it.current()->desktop_nr;
            bbtool->addFrameWindow(copy,desktop_it.current()->win,False);
            bbtool->windowList->insert(copy);
        }
    }
}

void WMInterface::removeSticky(Window win,int keep_on_desktop)
{
    int i;

    for (i=0;i<bbtool->getNumberOfDesktops();i++) {
        LinkedListIterator<WindowList> win_it(bbtool->windowList);
        for (; win_it.current(); win_it++) {
            if (win_it.current()->win==win &&
                win_it.current()->desktop_nr!=keep_on_desktop) {
                WindowList *old = win_it.current();
                bbtool->windowList->remove(old);
                XDestroyWindow(bbtool->getXDisplay(),old->pager_win);
                delete old;
            }
        }
    }
}

void WMInterface::changeIconState(Window win) {

    LinkedListIterator<WindowList> win_it(bbtool->windowList);
    for (; win_it.current(); win_it++)
        if ((win_it.current()->win) == win)
            break;

    if (win_it.current()) {
        WindowList *tmp = win_it.current();
    
        if (!tmp->icon) {
            if (!tmp->sticky) {
                XUnmapWindow(bbtool->getXDisplay(),tmp->pager_win);
                tmp->icon=True;
            } else {
                LinkedListIterator<WindowList> win_it(bbtool->windowList);
                for (; win_it.current(); win_it++) {
                    if (win_it.current()->win==tmp->win) {
                        XUnmapWindow(bbtool->getXDisplay(),win_it.current()->pager_win);
                        win_it.current()->icon=True;
                    }
                }
            }
        } else  {
            if (!tmp->sticky) {
                XMapWindow(bbtool->getXDisplay(),tmp->pager_win);
                tmp->icon=False;
            } else {
                LinkedListIterator<WindowList> win_it(bbtool->windowList);
                for (; win_it.current(); win_it++) {
                    if (win_it.current()->win==tmp->win) {
                        XMapWindow(bbtool->getXDisplay(),win_it.current()->pager_win);
                        win_it.current()->icon=False;
                    }
                }
            }
        }
    }
}

void WMInterface::windowAttributeChange(Window win) {
#ifdef DEBUG
    fprintf(stderr,"windowAttribute Change called\n");
#endif
}


void WMInterface::WMNotifyStartup() {
}

void WMInterface::WMNotifyWindowAdd(Window win,int desktop_nr) {
    if (hasWindowAtom(win,bbtool->getNETWMStateSkipPagerAtom()) ||
        isIconicState(win)) {
        return;
    }
    
    WindowList *tmp = new WindowList;
    tmp->win= win;
    int status=bbtool->getWindowGeometry(tmp);
    if (status) {
        tmp->icon=False;
        tmp->sticky=False;
        tmp->focused=hasFocus(win);
        tmp->shaded=hasWindowAtom(win,bbtool->getNETWMStateShadedAtom());
        tmp->desktop_nr=getWindowWorkspaceID(win);
        if (tmp->desktop_nr>=0) {
            bbtool->addFrameWindow(tmp,0,False);
            bbtool->windowList->insert(tmp);
            if (hasWindowAtom(win,bbtool->getNETWMStateStickyAtom()))
                addSticky(tmp);
        }
        else delete tmp;
    } else delete tmp;
  
}

void WMInterface::WMNotifyDel(Window win) {
    bbtool->removeWindow(win);
}

void WMInterface::WMNotifyAttributes(Window win) {
    windowAttributeChange(win);
}

void WMInterface::WMNotifyFocus(Window win) {
    if (bbtool->getResource()->getFocusStyle()!=none)
        bbtool->focusWindow(win);
}

void WMInterface::WMNotifyCurrentWorkspace(int current_desktop) {
    bbtool->desktopChange(current_desktop);
//      bbtool->redraw=True;
}

void WMInterface::WMNotifyWindowRaise(Window win) {
    bbtool->raiseWindow(win);
}

void WMInterface::WMNotifyWindowLower(Window win) {
    bbtool->lowerWindow(win);
}

void WMInterface::WMNotifyWorkspaceCount(int number_of_desktops) {
    int old_number_of_desktops = bbtool->getNumberOfDesktops();

    if (number_of_desktops < 0) return;
    bbtool->setNumberOfDesktops(number_of_desktops);
    if (number_of_desktops>old_number_of_desktops) {
        int i;
        for (i=old_number_of_desktops;i<number_of_desktops;i++) {
            DesktopList *tmp = new DesktopList;
            bbtool->addDesktopWindow(tmp,False);
            bbtool->desktopList->insert(tmp);
            LinkedListIterator<WindowList> win_it(bbtool->windowList);
            for (; win_it.current(); win_it++)
                if ((win_it.current()->sticky &&win_it.current()->desktop_nr==1))
                    break;
            if (win_it.current()) {
                WindowList *sticky= win_it.current();
                WindowList *copy = new WindowList;
                copy->win= sticky->win;
                copy->width=sticky->width;
                copy->height=sticky->height;
                copy->x_position=sticky->x_position;
                copy->y_position=sticky->y_position;
                copy->icon=sticky->icon;
                copy->sticky=sticky->sticky;
                copy->shaded=sticky->shaded;
                copy->desktop_nr=tmp->desktop_nr;
                bbtool->addFrameWindow(copy,0,False);
                bbtool->windowList->insert(copy);
            }
        }
    } else if (number_of_desktops<old_number_of_desktops)
        bbtool->removeDesktopWindow();
}

void WMInterface::handleWMEvents(XEvent Event) {
    fprintf(stderr,"handleBlackBoxEvents called\n");
}

void WMInterface::timeout(void) {
    bbtool->pollForChanges();
}
