// BaseDisplay.hh from Blackbox - an X11 Window manager
//
// Copyright (c) 2008 Mark Willson, mark@hydrus.org.uk
// Copyright (c) 1997 - 2000 Brad Hughes (bhughes@tcac.net)
//
// Permission is hereby granted, free of charge, to any person obtaining a
// copy of this software and associated documentation files (the "Software"),
// to deal in the Software without restriction, including without limitation
// the rights to use, copy, modify, merge, publish, distribute, sublicense,
// and/or sell copies of the Software, and to permit persons to whom the
// Software is furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
// THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
// FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
// DEALINGS IN THE SOFTWARE.


#ifndef   __BaseDisplay_hh
#define   __BaseDisplay_hh

#include <X11/Xlib.h>
#include <X11/Xatom.h>

// forward declaration
class BaseDisplay;
class ScreenInfo;

#include "LinkedList.hh"
#include "Timer.hh"

#ifndef    __EMX__
void bexec(const char *, char *);
#endif // !__EMX__

char *bstrdup(const char *);
template <typename Z> inline Z min(Z a, Z b) { return ((a < b) ? a : b); }
template <typename Z> inline Z max(Z a, Z b) { return ((a > b) ? a : b); }

class BaseDisplay {
private:
    struct cursor {
        Cursor session, move, ll_angle, lr_angle;
    } cursor;

    struct shape {
        Bool extensions;
        int event_basep, error_basep;
    } shape;

    Atom xa_wm_colormap_windows, xa_wm_protocols, xa_wm_state,
        xa_wm_delete_window, xa_wm_take_focus, xa_wm_change_state,
        motif_wm_hints;

    // root window properties
    Atom net_supported, net_client_list, net_client_list_stacking,
        net_number_of_desktops, net_desktop_geometry, net_desktop_viewport,
        net_current_desktop, net_desktop_names, net_active_window, net_workarea,
        net_supporting_wm_check, net_virtual_roots;

    // root window messages
    Atom net_close_window, net_wm_moveresize;

    // application window properties
    Atom net_properties, net_wm_name, net_wm_desktop, net_wm_window_type,
        net_wm_state, net_wm_strut, net_wm_icon_geometry, net_wm_icon, net_wm_pid,
        net_wm_handled_icons;

    // application protocols
    Atom net_wm_ping;

    // Additions by MPW
    Atom net_wm_state_skip_taskbar,net_wm_state_skip_pager,net_wm_state_sticky,
        net_wm_state_hidden,net_wm_state_shaded,
        net_wm_state_below,net_wm_window_type_dock;
    
    Bool _startup, _shutdown;
    Display *display;
    LinkedList<ScreenInfo> *screenInfoList;
    LinkedList<BTimer> *timerList;

    char *display_name, *application_name;
    int number_of_screens, server_grabs, colors_per_channel;


protected:
    // pure virtual function... you must override this
    virtual void process_event(XEvent *) = 0;


public:
    BaseDisplay(char *, char * = 0);
    virtual ~BaseDisplay(void);

    inline const Atom &getWMChangeStateAtom(void) const
    { return xa_wm_change_state; }
    inline const Atom &getWMStateAtom(void) const
    { return xa_wm_state; }
    inline const Atom &getWMDeleteAtom(void) const
    { return xa_wm_delete_window; }
    inline const Atom &getWMProtocolsAtom(void) const
    { return xa_wm_protocols; }
    inline const Atom &getWMTakeFocusAtom(void) const
    { return xa_wm_take_focus; }
    inline const Atom &getWMColormapAtom(void) const
    { return xa_wm_colormap_windows; }
    inline const Atom &getMotifWMHintsAtom(void) const
    { return motif_wm_hints; }

    // root window properties
    inline const Atom &getNETSupportedAtom(void) const
    { return net_supported; }
    inline const Atom &getNETClientListAtom(void) const
    { return net_client_list; }
    inline const Atom &getNETClientListStackingAtom(void) const
    { return net_client_list_stacking; }
    inline const Atom &getNETNumberOfDesktopsAtom(void) const
    { return net_number_of_desktops; }
    inline const Atom &getNETDesktopGeometryAtom(void) const
    { return net_desktop_geometry; }
    inline const Atom &getNETDesktopViewportAtom(void) const
    { return net_desktop_viewport; }
    inline const Atom &getNETCurrentDesktopAtom(void) const
    { return net_current_desktop; }
    inline const Atom &getNETDesktopNamesAtom(void) const
    { return net_desktop_names; }
    inline const Atom &getNETActiveWindowAtom(void) const
    { return net_active_window; }
    inline const Atom &getNETWorkareaAtom(void) const
    { return net_workarea; }
    inline const Atom &getNETSupportingWMCheckAtom(void) const
    { return net_supporting_wm_check; }
    inline const Atom &getNETVirtualRootsAtom(void) const
    { return net_virtual_roots; }

    // root window messages
    inline const Atom &getNETCloseWindowAtom(void) const
    { return net_close_window; }
    inline const Atom &getNETWMMoveResizeAtom(void) const
    { return net_wm_moveresize; }

    // application window properties
    inline const Atom &getNETPropertiesAtom(void) const
    { return net_properties; }
    inline const Atom &getNETWMNameAtom(void) const
    { return net_wm_name; }
    inline const Atom &getNETWMDesktopAtom(void) const
    { return net_wm_desktop; }
    inline const Atom &getNETWMWindowTypeAtom(void) const
    { return net_wm_window_type; }
    inline const Atom &getNETWMStateAtom(void) const
    { return net_wm_state; }
    inline const Atom &getNETWMStrutAtom(void) const
    { return net_wm_strut; }
    inline const Atom &getNETWMIconGeometryAtom(void) const
    { return net_wm_icon_geometry; }
    inline const Atom &getNETWMIconAtom(void) const
    { return net_wm_icon; }
    inline const Atom &getNETWMPidAtom(void) const
    { return net_wm_pid; }
    inline const Atom &getNETWMHandledIconsAtom(void) const
    { return net_wm_handled_icons; }

    // application protocols
    inline const Atom &getNETWMPingAtom(void) const
    { return net_wm_ping; }
    // for fluxbox 1.x.x
    inline const Atom &getNETWMStateSkipTaskbarAtom(void) const
    { return net_wm_state_skip_taskbar; }
    inline const Atom &getNETWMStateSkipPagerAtom(void) const
    { return net_wm_state_skip_pager; }
    inline const Atom &getNETWMStateStickyAtom(void) const
    { return net_wm_state_sticky; }
    inline const Atom &getNETWMStateHiddenAtom(void) const
    { return net_wm_state_hidden; }
    inline const Atom &getNETWMStateShadedAtom(void) const
    { return net_wm_state_shaded; }
    inline const Atom &getNETWMStateBelowAtom(void) const
    { return net_wm_state_below; }
    inline const Atom &getNETWMWindowTypeDockAtom(void) const
    { return net_wm_window_type_dock; }

    inline ScreenInfo *getScreenInfo(int s)
    { return (ScreenInfo *) screenInfoList->find(s); }

    inline const Bool &hasShapeExtensions(void) const
    { return shape.extensions; }
    inline const Bool &doShutdown(void) const
    { return _shutdown; }
    inline const Bool &isStartup(void) const
    { return _startup; }

    inline const Cursor &getSessionCursor(void) const
    { return cursor.session; }
    inline const Cursor &getMoveCursor(void) const
    { return cursor.move; }
    inline const Cursor &getLowerLeftAngleCursor(void) const
    { return cursor.ll_angle; }
    inline const Cursor &getLowerRightAngleCursor(void) const
    { return cursor.lr_angle; }

    inline Display *getXDisplay(void) { return display; }

    inline const char *getXDisplayName(void) const
    { return (const char *) display_name; }
    inline const char *getApplicationName(void) const
    { return (const char *) application_name; }

    inline const int &getNumberOfScreens(void) const
    { return number_of_screens; }
    inline const int &getShapeEventBase(void) const
    { return shape.event_basep; }

    inline void shutdown(void) { _shutdown = True; }
    inline void run(void) { _startup = _shutdown = False; }

    const Bool validateWindow(Window);

    void grab(void);
    void ungrab(void);
    void eventLoop(void);
    void addTimer(BTimer *);
    void removeTimer(BTimer *);

    // another pure virtual... this is used to handle signals that BaseDisplay
    // doesn't understand itself
    virtual Bool handleSignal(int) = 0;
};


class ScreenInfo {
private:
    BaseDisplay *basedisplay;
    Visual *visual;
    Window root_window;

    int depth, screen_number;
    unsigned int width, height;


protected:


public:
    ScreenInfo(BaseDisplay *, int);

    inline BaseDisplay *getBaseDisplay(void) { return basedisplay; }
    inline Visual *getVisual(void) { return visual; }
    inline const Window &getRootWindow(void) const { return root_window; }

    inline const int &getDepth(void) const { return depth; }
    inline const int &getScreenNumber(void) const { return screen_number; }

    inline const unsigned int &getWidth(void) const { return width; }
    inline const unsigned int &getHeight(void) const { return height; }
};


#endif // __BaseDisplay_hh
